unit DemoDlg;

interface

uses
  Winapi.Windows, Winapi.Messages, Winapi.ShlwApi, Winapi.ShellApi, System.Types, System.SysUtils, System.Variants, System.Classes, Vcl.Graphics,
  Vcl.Controls, Vcl.Forms, Vcl.Dialogs, Vcl.ExtCtrls, Vcl.StdCtrls, ScnLib;

type
  TMainWindow = class(TForm)
    screen_capture: TGroupBox;
    full_screen: TRadioButton;
    a_region_window: TRadioButton;
    pc_game_screen: TRadioButton;
    audio_capture: TGroupBox;
    playback_audio: TCheckBox;
    microphone_audio: TCheckBox;
    webcam_capture: TGroupBox;
    webcam_preview: TCheckBox;
    webcam_overlay: TCheckBox;
    full_webcam_video: TCheckBox;
    output_video: TGroupBox;
    video_path: TEdit;
    change_video_path: TButton;
    browse_video_path: TButton;
    rec_time: TLabel;
    start_stop: TButton;
    pause_resume: TButton;
    more_settings: TButton;
    timer_REC: TTimer;
    timer_FPS: TTimer;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure timer_RECTimer(Sender: TObject);
    procedure timer_FPSTimer(Sender: TObject);
    procedure start_stopClick(Sender: TObject);
    procedure pause_resumeClick(Sender: TObject);
    procedure more_settingsClick(Sender: TObject);
    procedure change_video_pathClick(Sender: TObject);
    procedure browse_video_pathClick(Sender: TObject);
    procedure full_screenClick(Sender: TObject);
    procedure a_region_windowClick(Sender: TObject);
    procedure pc_game_screenClick(Sender: TObject);
    procedure playback_audioClick(Sender: TObject);
    procedure microphone_audioClick(Sender: TObject);
    procedure webcam_previewClick(Sender: TObject);
    procedure webcam_overlayClick(Sender: TObject);
    procedure full_webcam_videoClick(Sender: TObject);
  private
    { Private declarations }
    m_dibFPS: TBitmap;
    procedure Init_Game_FPS_Overlay();
    procedure Free_Game_FPS_Overlay();
    procedure Update_Game_FPS_Overlay();
    procedure Update_UI();
    procedure Start_Recording();
    procedure Stop_Recording(bSuccessful: Boolean);
  public
    { Public declarations }
  end;

var
  MainWindow: TMainWindow;

implementation

{$R *.dfm}

procedure TMainWindow.FormCreate(Sender: TObject);
var
  logPath: array[0..MAX_PATH] of WideChar;
begin
	// Make the log file path the same as the process exe file path but different extension.
	GetModuleFileName(0, logPath, MAX_PATH);
	PathRenameExtension(logPath, '.txt');

	// Create a log file that is helpful for debugging.
	// It can be called before calling ScnLib_Initialize().
	ScnLib_SetLogPathW(logPath, True);

	// Initialize the SDK.
	// It must be called before calling most of the other APIs of the SDK.
	// Usually it's called only once at the start of the program.
	ScnLib_InitializeW('');

	// Make the screen capture region frame visible. It's invisible by default.
	// If you don't like the style of the SDK built-in screen capture region frame, you may implement your own.
	ScnLib_ShowCaptureRegionFrame(True);

	// We prefer 1080p webcam resolution. It's 320x240 by default.
	// If your webcam doesn't support the resolution you set, the SDK will choose the closest resolution automatically.
	ScnLib_SetWebcamResolution(1920, 1080);

	// Default mode: Capture Full Screen
	full_screen.Checked := True;

	// Initialize the game FPS DIB resource.
	Init_Game_FPS_Overlay();

	// Setup a timer to update the game FPS overlay for game capture mode
	timer_FPS.Enabled := True;

	Update_UI();
end;

procedure TMainWindow.FormDestroy(Sender: TObject);
begin
	// Release the game FPS DIB resource.
	Free_Game_FPS_Overlay();

	// Uninitialize the SDK.
	// It must be called before exiting the process or there might be resource leak.
	// Usually it's called only once at the end of the program.
	ScnLib_Uninitialize();
end;

procedure TMainWindow.timer_RECTimer(Sender: TObject);
var
  szRecTime: array[0..11] of WideChar;
begin
	// Since the SDK doesn't have any callback mechanism,
	// you need to check the recording status periodically in a timer.
	// If it's recording then update the recording time info.
	if ScnLib_IsRecording() then
	begin
		// Get the current recording time string in the format of 'HH:MM:SS'.
		ScnLib_GetRecTimeW(szRecTime);

		rec_time.Caption := szRecTime;
	end
	// If it's not recording then stop recording on failure.
	// The recording may stop automatically if a fatal error occurs.
	// In this case you still need to do some cleanup and UI updates.
	else
	begin
		Stop_Recording(False);

		Update_UI();
	end;
end;

procedure TMainWindow.timer_FPSTimer(Sender: TObject);
begin
	// Is currently game capture mode and a Direct3D/OpenGL game window detected?
	if ScnLib_GetGameWnd() <> 0 then
	begin
		// Update the game FPS overlay.
		Update_Game_FPS_Overlay();
	end;
end;

procedure TMainWindow.start_stopClick(Sender: TObject);
begin
	if not ScnLib_IsRecording() then
	begin
		// Start recording if no recording is in progress.
		Start_Recording();
	end
	else
	begin
		// Stop recording if a recording is in progress.
		Stop_Recording(True);
	end;

	Update_UI();
end;

procedure TMainWindow.pause_resumeClick(Sender: TObject);
begin
	if not ScnLib_IsPaused() then
	begin
		// Pause recording if the recording is not paused.
		ScnLib_PauseRecording();
	end
	else
	begin
		// Resume recording if the recording is paused.
		ScnLib_ResumeRecording();
	end;

	Update_UI();
end;

procedure TMainWindow.more_settingsClick(Sender: TObject);
begin
	// Popup the SDK built-in settings dialog to let user configure more settings.
	// If you don't like the style of the SDK built-in settings dialog, you may implement your own.
	ScnLib_ConfigureSettings(Handle);

	Update_UI();
end;

procedure TMainWindow.change_video_pathClick(Sender: TObject);
var
  saveDlg: TSaveDialog;
begin
  saveDlg := TSaveDialog.Create(nil);

	// Supported video file formats are MP4, FLV and AVI.
	saveDlg.Filter := 'MP4 videos (*.mp4)|*.mp4|FLV videos (*.flv)|*.flv|AVI videos (*.avi)|*.avi';
  saveDlg.DefaultExt := 'mp4';
  saveDlg.Title := 'Set Output Video File Path';
  saveDlg.Options := saveDlg.Options + [ofOverwritePrompt];

	// Popup a save file dialog to let user change the output video file path.
	if saveDlg.Execute(Handle) then
	begin
		// Set the output video file path to be created.
		ScnLib_SetVideoPathW(PWideChar(saveDlg.FileName));

		Update_UI();
	end;
end;

procedure TMainWindow.browse_video_pathClick(Sender: TObject);
var
  videoPath: array[0..MAX_PATH] of WideChar;
  filePath: string;
begin
	// Get the output video file path to be created.
	ScnLib_GetVideoPathW(videoPath, False);

	filePath := videoPath;

	// Remove the invalid file path chars that are used by SDK-defined variables: <num>, <date> and <time>
	filePath := StringReplace(filePath, '<', '', [rfReplaceAll]);
	filePath := StringReplace(filePath, '>', '', [rfReplaceAll]);

	// Get the folder where your recording videos are saved.
	StrCpy(videoPath, PWideChar(filePath));
	PathRemoveFileSpec(videoPath);

	if PathIsDirectory(videoPath) then
	begin
		// Browse the video folder in the Windows File Explorer.
		ShellExecute(Handle, nil, 'explorer.exe', videoPath, nil, SW_SHOWNORMAL);
	end;
end;

procedure TMainWindow.full_screenClick(Sender: TObject);
begin
	// Don't bind any capture window.
	ScnLib_SetCaptureWnd(0, False);
	// Set all-zero coordinates to let SDK detect and use the full screen coordinates.
	ScnLib_SetCaptureRegion(0, 0, 0, 0);
	// Make sure the game capture mode is disabled.
	ScnLib_EnableGameCaptureMode(False);
end;

procedure TMainWindow.a_region_windowClick(Sender: TObject);
var
  hwnd: THandle;
  l, t, r, b: Integer;
begin
	// Turn mouse cursor into a crosshair and let user select a region/window on screen.
	// To select a region:
	//   1) Move the mouse cursor to the top-left corner of the region you want to capture.
	//   2) Press and hold the left mouse button, and then move the mouse cursor to the bottom-right corner of the region you want to capture.
	//   3) Release the left mouse button, done. Or, you can click the right mouse button to cancel in the middle of the process.
	// To select a window:
	//   1) Move the mouse cursor over the window you want to capture and then the window will be highlighted.
	//   2) Click the left mouse button to select the pointing window.
	// To cancel selection, click the right mouse button.
	// If a region is selected, you will get the region coordinates and a zero window handle.
	// If a window is selected, you will get the window coordinates and a non-zero window handle.
	if ScnLib_SelectCaptureRegionW(&l, &t, &r, &b, &hwnd, '') then
	begin
		// Bind the capture window if hwnd is non-zero.
		// Or unbind the capture window if hwnd is zero.
		ScnLib_SetCaptureWnd(hwnd, True);
		// Set the capture region coordinates.
		ScnLib_SetCaptureRegion(l, t, r, b);
	end;

	// Make sure the game capture mode is disabled.
	ScnLib_EnableGameCaptureMode(False);
end;

procedure TMainWindow.pc_game_screenClick(Sender: TObject);
begin
	// Don't bind any capture window. (You can bind a specific game window if you don't want SDK to detect foreground game window automatically)
	ScnLib_SetCaptureWnd(0, False);
	// Set all-zero coordinates to let SDK detect and use the full screen coordinates if no game screen is detected.
	ScnLib_SetCaptureRegion(0, 0, 0, 0);
	// Enable the game capture mode.
	ScnLib_EnableGameCaptureMode(True);
end;

procedure TMainWindow.playback_audioClick(Sender: TObject);
begin
	// Enable/Disable capture from the playback audio source (speakers / headphone).
	ScnLib_RecordAudioSource(True, playback_audio.Checked);
end;

procedure TMainWindow.microphone_audioClick(Sender: TObject);
begin
	// Enable/Disable capture from the recording audio source (microphone / line-in).
	ScnLib_RecordAudioSource(False, microphone_audio.Checked);
end;

procedure TMainWindow.webcam_previewClick(Sender: TObject);
begin
	// Open/Close the webcam preview window.
	ScnLib_PreviewWebcam(webcam_preview.Checked, 0, True, 0);
end;

procedure TMainWindow.webcam_overlayClick(Sender: TObject);
begin
	if webcam_overlay.Checked then
	begin
		// Enable webcam capture by selecting the first webcam device.
		ScnLib_SelectWebcamDevice(0);
		// Don't record webcam only. Put the webcam overlay on the screen capture video.
		ScnLib_RecordWebcamOnly(False);
	end
	else if not full_webcam_video.Checked then
	begin
		// Close the webcam preview window.
		ScnLib_PreviewWebcam(False, 0, True, 0);
		// Disable webcam capture by selecting a negative index.
		ScnLib_SelectWebcamDevice(-1);
	end;

	Update_UI();
end;

procedure TMainWindow.full_webcam_videoClick(Sender: TObject);
begin
	if full_webcam_video.Checked then
	begin
		// Enable webcam capture by selecting the first webcam device.
		ScnLib_SelectWebcamDevice(0);
		// Record webcam only. Discard screen capture. Make a full webcam video.
		ScnLib_RecordWebcamOnly(True);
	end
	else if not webcam_overlay.Checked then
	begin
		// Close the webcam preview window.
		ScnLib_PreviewWebcam(False, 0, True, 0);
		// Disable webcam capture by selecting a negative index.
		ScnLib_SelectWebcamDevice(-1);
	end;

	Update_UI();
end;

procedure TMainWindow.Init_Game_FPS_Overlay();
begin
	// FPS: Frames Per Second
	// DIB: Device-Independent Bitmap
  // Create the game FPS DIB.
  m_dibFPS := TBitmap.Create;
  m_dibFPS.HandleType := bmDIB;
  m_dibFPS.PixelFormat := pf24bit;
  m_dibFPS.Width := 170;
  m_dibFPS.Height := -70; // a top-down bitmap
  m_dibFPS.Canvas.Font.Name := 'Impact';
end;

procedure TMainWindow.Free_Game_FPS_Overlay();
begin
  m_dibFPS.Free;
end;

procedure TMainWindow.Update_Game_FPS_Overlay();
var
  w, h: Integer;
  rc: TRect;
  strFPS: string;
  szRecTime: array[0..11] of WideChar;
begin
	// The game FPS DIB dimension.
  w := m_dibFPS.Width;
  h := m_dibFPS.Height;
  SetRect(&rc, 0, 0, w, h);

	// Get the game FPS number.
	strFPS := Format('%d fps', [ScnLib_GetGameFrameRate()]);

	// Get the current recording time string in the format of 'HH:MM:SS'.
	ScnLib_GetRecTimeW(szRecTime);

	// Clear the overlay background.
  m_dibFPS.Canvas.Brush.Style := bsSolid;
  m_dibFPS.Canvas.Brush.Color := TColor(RGB(255, 255, 255));
  m_dibFPS.Canvas.FillRect(rc);

	// Shrink the drawing area.
  rc.Inflate(-3, -3);

	// Display a camera icon on the left side.
  DrawIconEx(m_dibFPS.Canvas.Handle, rc.left, rc.top, Icon.Handle, rc.Height, rc.Height, 0, 0, DI_NORMAL);

	// Draw the game FPS number.
  m_dibFPS.Canvas.Font.Height := -30;
	m_dibFPS.Canvas.Font.Color := TColor(RGB(128, 128, 128));
	DrawText(m_dibFPS.Canvas.Handle, strFPS, -1, &rc, DT_RIGHT or DT_TOP or DT_SINGLELINE);

	// Draw the REC time.
  m_dibFPS.Canvas.Font.Height := -25;
  if ScnLib_IsRecording() then
  	m_dibFPS.Canvas.Font.Color := TColor(RGB(255, 0, 0))
  else
  	m_dibFPS.Canvas.Font.Color := TColor(RGB(0, 200, 0));
	DrawText(m_dibFPS.Canvas.Handle, szRecTime, -1, &rc, DT_RIGHT or DT_BOTTOM or DT_SINGLELINE);

	// Show the game FPS overlay on the top-right corner of the game screen.
	ScnLib_SetInGameOverlayPosition(POSITION_TOP_RIGHT, 5, 5);
	ScnLib_ShowInGameOverlay(m_dibFPS.ScanLine[h - 1], w, h, 24);
end;

procedure TMainWindow.Update_UI();
var
  videoPath: array[0..MAX_PATH] of WideChar;
begin
	// Update the radio boxes in the Screen Capture group.
	full_screen.Enabled := not ScnLib_IsRecording();
	a_region_window.Enabled := not ScnLib_IsRecording();
	pc_game_screen.Enabled := not ScnLib_IsRecording();

	// Update the check boxes in the Audio Capture group.
	playback_audio.Checked := ScnLib_IsRecordAudioSource(True); // True - the playback audio source (speakers / headphone)
	microphone_audio.Checked := ScnLib_IsRecordAudioSource(False); // False - the recording audio source (microphone / line-in)

	// Update the check boxes in the Webcam Capture group
	// The webcam capture is enabled only when a webcam device is selected.
	// The selected webcam device index is a zero-based number. The first webcam device index is 0.
	if ScnLib_GetSelectedWebcamDevice() >= 0 then
	begin
		webcam_preview.Enabled := True;

		// If the webcam is being previewed you can get a non-zero preview window handle.
		if ScnLib_GetWebcamPreviewWnd() <> 0 then
		begin
			webcam_preview.Checked := True;
		end
		// Otherwise the webcam is not being previewed.
		else
		begin
			webcam_preview.Checked := False;
		end;

		// If it's set to record webcam only, screen capture will be discarded and you will get a full webcam video.
		if ScnLib_IsRecordWebcamOnly() then
		begin
			webcam_overlay.Checked := False;
			full_webcam_video.Checked := True;
		end
		// Otherwise you will get a screen recording video with a webcam overlay on it.
		else
		begin
			webcam_overlay.Checked := True;
			full_webcam_video.Checked := False;
		end;
	end
	// A negative index means no webcam device is selected.
	else
	begin
		webcam_preview.Enabled := False;
		webcam_preview.Checked := False;
		webcam_overlay.Checked := False;
		full_webcam_video.Checked := False;
	end;

	// Get the output video file path to be created.
	ScnLib_GetVideoPathW(videoPath, False);

	// Update the elements in the Output Video group.
	video_path.Text := videoPath;
	video_path.Enabled := not ScnLib_IsRecording();
	change_video_path.Enabled := not ScnLib_IsRecording();

	// Update the recording control buttons at the bottom.
	if ScnLib_IsRecording() then
		start_stop.Caption := 'Stop'
	else
		start_stop.Caption := 'Start';

	if ScnLib_IsPaused() then
		pause_resume.Caption := 'Resume'
	else
		pause_resume.Caption := 'Pause';

	pause_resume.Enabled := ScnLib_IsRecording();
end;

procedure TMainWindow.Start_Recording();
begin
	// If no foreground Direct3D/OpenGL game window is detected, prompt user what to do.
	if ScnLib_IsGameCaptureModeEnabled() and (ScnLib_GetGameWnd() = 0) then
	begin
		MessageBox(Handle, 'No foreground Direct3D/OpenGL game window is detected!'#13#10#13#10 +
'Please open a Direct3D/OpenGL game and place its window in the foreground, ' +
'or the SDK will automatically fallback to standard screen capture mode.',
			PWideChar(Caption), MB_ICONINFORMATION or MB_OK);
	end;

	// Set the output video file path to be created.
	// The file path can include SDK-defined variables: <num>, <date> and <time>
	ScnLib_SetVideoPathW(PWideChar(video_path.Text));

	// Start recording now.
	if ScnLib_StartRecording() then
	begin
		// Start the timer to check the recording status and update recording time.
		timer_REC.Enabled := True;
	end
	else
	begin
		// Do some cleanup if recording failed to start.
		Stop_Recording(False);
	end;
end;

procedure TMainWindow.Stop_Recording(bSuccessful: Boolean);
var
  filePath: array[0..MAX_PATH] of WideChar;
begin
	// Stop the timer
	timer_REC.Enabled := False;

	// Stop the recording. It's OK to call it even if no recording is in progress.
	ScnLib_StopRecording();

	if bSuccessful then
	begin
		// Get the saved video file path if the recording is done successfully.
		ScnLib_GetVideoPathW(filePath, True);
	end
	else
	begin
		// Get the saved log file path if the recording is failed.
		ScnLib_GetLogPathW(filePath);
	end;

	// Play the video file or show the log file.
	if PathFileExists(filePath) then
	begin
		ShellExecute(Handle, 'open', filePath, nil, nil, SW_SHOWNORMAL);
	end;
end;

end.
